<?php
// Define the plugin namespace and class
namespace CBM;

class CloneBackupMigratePlugin {

	// Constructor
	public function __construct() {
		// Register the 'cbm-backup' custom post type
		add_action('init', array($this, 'registerBackupPostType'));

		//Register a callback function that will be triggered on plugin activation
		register_activation_hook(__DIR__ . '/cbm.php', array($this, 'activatePlugin'));

		// Add necessary WordPress action hooks and filters
		add_action('admin_menu', array($this, 'addPluginMenu'));
		add_action('admin_enqueue_scripts', array($this, 'enqueueScripts'));

		// Add action hook for handling download of archive file for cbm-post
		add_action('template_redirect', array($this, 'handleDownloadArchive'));
		
		// Add more necessary hooks and filters as required
	}

	public function activatePlugin() {
		$this->registerBackupPostType();
		flush_rewrite_rules(false);
	}

	// Method to register the 'cbm-backup' custom post type
	public function registerBackupPostType() {
		register_post_type('cbm-backup',
			array(
				'labels' => array(
					'name' => __('Backups', 'cbm'),
					'singular_name' => __('Backup', 'cbm'),
				),
				'public' => true,
				'has_archive' => false,
				'supports' => array('title', 'editor'),
			)
		);
	}

	// Method to add plugin menu in the WordPress admin menu
	public function addPluginMenu() {
		add_menu_page(
			'Clone, Backup & Migrate',
			'Clone, Backup & Migrate',
			'manage_options',
			'clone-backup-migrate',
			array($this, 'pluginSettingsPage')
		);
	}

	// Method to enqueue necessary scripts and styles
	public function enqueueScripts() {
		//wp_enqueue_script('cbm-script', plugins_url('/js/script.js', __FILE__), array('jquery'), '1.0.0', true);
		//wp_enqueue_style('cbm-style', plugins_url('/css/style.css', __FILE__), array(), '1.0.0');
	}

	// Method to render the plugin settings page
	public function pluginSettingsPage() {
		// Get the list of existing backups
		$backups = get_posts(array('post_type' => 'cbm-backup'));
			
		// Check if the form for creating a new backup is submitted
		if (isset($_POST['create_backup'])) {
			$this->handleBackup();
		}

		if (isset($_POST['import_backup'])) {
			$backupUrl = $_POST['backup_url'];
			$this->handleImportBackup($backupUrl);
		}

		// Check if the form for restoring a backup is submitted
		if (isset($_POST['restore_backup'])) {
			$backupPostId = $_POST['backup_post_id'];
			$this->handleRestore($backupPostId);
		}

		// Check if the form for deleting a backup is submitted
		if (isset($_POST['delete_backup'])) {
			$backupPostId = $_POST['backup_post_id'];
			$this->deleteBackup($backupPostId);
		}

		$current = ( $_GET['tab'] ) ?? 'backup';
		
		?>
		<div class="wrap">
			<h1>Clone, Backup & Migrate Settings</h1>
			
			<!-- Display the HTML markup for the plugin settings page -->
			<?php $this->displaySettingsPage($current); ?>

			<?php /*<!-- Autobackup tab content -->
			<div id="autobackup" class="tab-content">
				<h2>Import</h2>
				<!--<h2>Automatic Backup Settings</h2>
				<form method="post" action="">
					<p><label><input type="checkbox" name="enable_autobackup" value="1"> Enable Automatic Backups</label></p>
					<p>
						<label>Backup Delay (in days):
							<input type="number" name="backup_delay" min="1" step="1" value="7">
						</label>
					</p>
					<p>
						<label>Number of Simultaneous Backups to Keep:
							<input type="number" name="simultaneous_backups" min="1" max="7" step="1" value="5">
						</label>
					</p>
					<button type="submit" name="save_autobackup" class="button button-primary">Save Settings</button>
				</form>-->
			</div>

			<!-- Third tab content -->
			<div id="thirdtab" class="tab-content">
				<h2>Export</h2>
				<!-- Content for the third tab goes here -->
			</div>*/ ?>
		</div>
	<?php
	}

	private function displaySettingsPage($current = 'backup') {
		$tabs = array(
			'backup' => 'Backup',
			'import' => 'Import',
			'export' => 'Export'
		);
		?>

		<!-- Create the tabs -->
		<h2 class="nav-tab-wrapper">
			<?php foreach( $tabs as $tab => $name ) {
				$class = ( $tab == $current ) ? ' nav-tab-active' : '';
				echo "<a class='nav-tab$class' href='?page=clone-backup-migrate&tab=$tab'>$name</a>";

			} ?>
		</h2>
		<div class="tabcontent">
			<?php if ($current === 'backup') {
				$this->displayBackupTabSettingsPage();
			} else if ($current === 'import') {
				$this->displayImportTabSettingsPage();
			} ?>
		</div>
		<?php
	}

	private function displayBackupTabSettingsPage() {
		$backups = get_posts(array('post_type' => 'cbm-backup')); ?>
		
		<!-- Display the list of existing backups -->
		<?php if (!empty($backups)) { ?>
			<table class="wp-list-table widefat fixed striped">
				<thead>
					<tr>
						<th>Backup Date</th>
						<th>Actions</th>
						<th>Download</th>
						<th>Delete</th>
					</tr>
				</thead>
				<tbody>
					<?php foreach ($backups as $backup) { ?>
						<tr>
							<td><?php echo $backup->post_title; ?></td>
							<td>
								<form method="post" action="">
									<input type="hidden" name="backup_post_id" value="<?php echo $backup->ID; ?>">
									<button type="submit" name="restore_backup" class="button">Restore</button>
								</form>
							</td>
							<td>
								<a href="<?php echo get_permalink($backup->ID); ?>" target="_blank">
									<button type="submit" name="download_backup" class="button">Download</button>
								</a>
							</td>
							<td>
								<form method="post" action="">
									<input type="hidden" name="backup_post_id" value="<?php echo $backup->ID; ?>">
									<button type="submit" name="delete_backup" class="button">Delete</button>
								</form>
							</td>
						</tr>
					<?php } ?>
				</tbody>
			</table>
		<?php } else { ?>
			<p>No backups found.</p>
		<?php } ?>
		<form method="post" action="">
			<button type="submit" name="create_backup" class="button button-primary">Create Backup</button>
		</form>
		<?php
	}

	private function displayImportTabSettingsPage() {
		?>
		<form method="post" action="">
			<p>
				<label for="backup_url">Backup Archive URL:</label>
				<input type="text" id="backup_url" name="backup_url" required>
			</p>
			<button type="submit" name="import_backup" class="button button-primary">Import Archive</button>
		</form>
		<?php
	}

	public function handleDownloadArchive() {
		// Check if the post type is cbm-post
		if (get_post_type() === 'cbm-backup') {
			// Check if the current user is an administrator
			if (current_user_can('administrator')) {
				$postId = get_the_id();

				// Retrieve the archive file path from the post content
				$archiveFilePath = get_post($postId)->post_content;

				// Check if the archive file path exists
				if (file_exists($archiveFilePath)) {
					// Set the appropriate content headers
					header('Content-Type: application/octet-stream');
					header('Content-Disposition: attachment; filename=' . basename($archiveFilePath));
					header('Content-Length: ' . filesize($archiveFilePath));

					// Output the archive file
					readfile($archiveFilePath);
				}
			} else {
				global $wp_query;
				$wp_query->set_404();
				status_header( 404 );
				get_template_part( 404 );
			}
		}
	}

	// Method to handle importing a backup archive
	public function handleImportBackup($backupUrl) {
		set_time_limit(0);
		
		// Get the backup location from plugin settings
		$backupLocation = get_option('cbm_backup_location');
		
		// Generate a unique backup filename
		$backupFilename = 'backup_' . date('Ymd_His') . '.zip';
		$backupPath = $backupLocation . '/' . $backupFilename;
		
		// Download the backup archive from the specified URL
		$result = $this->downloadBackupArchive($backupUrl, $backupPath);
		
		// Check if the download was successful
		if ($result) {
			// Create a backup post for the imported archive
			$this->createBackupPost($backupPath);
			
			// Display success message or perform any additional actions
			echo 'Backup imported successfully!';
		} else {
			// Display error message or perform any error handling
			echo 'Failed to import backup!';
		}
	}

	// Method to download a backup archive from a URL
	public function downloadBackupArchive($backupUrl, $backupPath) {
		// Use WordPress HTTP API to download the backup archive from the specified URL
		$response = wp_safe_remote_get($backupUrl);

		// Check if the download was successful
		if (!is_wp_error($response) && $response['response']['code'] == 200) {
			// Save the backup archive to the specified destination
			if (file_put_contents($backupPath, $response['body']) !== false) {
				return true;
			}
		}

		return false;
	}

	// Method to handle backup functionality
	public function handleBackup() {
		set_time_limit(0);
		$time = time();

		// Code to create a backup archive including the database and content
		// Get the backup location from plugin settings
		$backupLocation = get_option('cbm_backup_location');

		// Generate a unique backup filename
		$backupFilename = 'backup_' . date('Ymd_His', $time) . '.zip';
		$backupPath = $backupLocation . '/' . $backupFilename;

		// Create a backup folder if it doesn't exist
		if (!file_exists($backupLocation)) {
			mkdir($backupLocation, 0755, true);
		}

		// Create the backup archive
		$zip = new \ZipArchive();
		if ($zip->open($backupPath, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) === true) {
			// Backup the database
			$this->backupDatabase($zip);

			// Backup the WordPress content
			$this->backupContent($zip);

			// Close the archive
			$zip->close();

			$this->createBackupPost($backupPath, $time);

			// Display success message or perform any additional actions
			echo 'Backup created successfully!';
		} else {
			// Display error message or perform any error handling
			echo 'Failed to create backup!';
		}
	}

	// Method to backup the WordPress database
	private function backupDatabase($zip) {
		global $wpdb;
		
		$zip->addEmptyDir('database');

		// Get the tables in the WordPress database
		$tables = $wpdb->get_results("SHOW TABLES", ARRAY_N);
		
		// Backup each table individually
		foreach ($tables as $table) {
			$tableName = $table[0];
			
			// Get the export file for the table
			$exportFilePath = $this->exportTable($tableName);
			if ($exportFilePath !== false) {
				// Add the export file to the backup archive
				$zip->addFile($exportFilePath, 'database'.DIRECTORY_SEPARATOR.$tableName.'.sql');
			}
		}
	}

	// Method to export a table into a file
	private function exportTable($tableName) {
		global $wpdb;

		// Generate a unique filename for the export file
		//$exportFilename = 'table_' . $tableName . '_' . date('Ymd_His') . '.sql';
		$exportFilename = 'table_' . $tableName . '.sql';
		$exportFilePath = sys_get_temp_dir() . '/' . $exportFilename;

		// Get the export file handle
		$fileHandle = fopen($exportFilePath, 'w');

		// Get the table structure
		$tableStructure = $wpdb->get_row("SHOW CREATE TABLE {$tableName}");
		$tableStructureQuery = $tableStructure->{'Create Table'};

		// Write the table structure to the export file
		fwrite($fileHandle, $tableStructureQuery . ";\n");

		// Fetch all rows from the table
		$rows = $wpdb->get_results("SELECT * FROM $tableName", ARRAY_A);

		// Loop through each row and generate the SQL insert statement
		if (!empty($rows)) {
			foreach ($rows as $row) {
				// Escape values
				$escapedValues = array_map('esc_sql', $row);

				$columns = implode("`, `", array_keys($escapedValues));
				$values = implode("', '", $escapedValues);

				// Build the insert statement
				$insertStatement = "INSERT INTO `{$tableName}` (`{$columns}`) VALUES ('{$values}');\n";

				// Write the insert statement to the export file
				fwrite($fileHandle, $insertStatement);
			}
			/*foreach ($rows as $row) {
				// Escape values according to their data type
				$escapedValues = array();
				foreach ($row as $key => $value) {
					if (is_numeric($value)) {
						$escapedValues[$key] = $value;
					} else {
						$escapedValues[$key] = "'" . esc_sql($value) . "'";
					}
				}

				// Build the insert statement
				$insertStatement = "INSERT INTO {$tableName} VALUES (" . implode(", ", $escapedValues) . ");\n";

				// Write the insert statement to the export file
				fwrite($fileHandle, $insertStatement);
			}*/
		}

		// Close the export file handle
		fclose($fileHandle);
		
		//Return the export file path
		return $exportFilePath;
	}

	// Method to backup the WordPress content
	private function backupContent($zip) {
		// Get the path of the WordPress installation
		$wordpressPath = ABSPATH;

		// Add the wordpress folder to the backup archive
		$this->addFolderToZip($zip, $wordpressPath, 'wordpress');
	}

	function createBackupPost($backupFilePath, $time) {
		// Create a new backup post
		$backupPost = array(
			'post_title' => 'Backup ' . date('Y-m-d H:i:s', $time),
			'post_date' => date('Y-m-d H:i:s', $time),
			'post_content' => $backupFilePath,
			'post_type' => 'cbm-backup',
			'post_status' => 'publish'
		);
		
		// Insert the backup post into the database
		$backupPostId = wp_insert_post($backupPost);
		
		// Return the ID of the created post if successful, false otherwise
		if (!is_wp_error($backupPostId)) {
			return $backupPostId;
		} else {
			return false;
		}
	}

	public function deleteBackup($backupPostId) {
		// Get the backup archive file path from the backup post
		$backupArchivePath = get_post($backupPostId)->post_content;

		// Delete the backup post
		wp_delete_post($backupPostId, true);

		// Delete the backup archive file
		if (file_exists($backupArchivePath)) {
			unlink($backupArchivePath);
		}

		// Display success message or perform any additional actions
		echo 'Backup deleted successfully!';
	}

	// Method to recursively copy a directory
	private function copyDirectory($src, $dst) {
		$dir = opendir($src);
		if (!file_exists($dst)) {
			mkdir($dst, 0755, true);
		}
		while (($file = readdir($dir)) !== false) {
			if ($file !== '.' && $file !== '..') {
				if (is_dir($src . '/' . $file)) {
					$this->copyDirectory($src . '/' . $file, $dst . '/' . $file);
				} else {
					copy($src . '/' . $file, $dst . '/' . $file);
				}
			}
		}
		closedir($dir);
	}

	private function addFolderToZip($zip, $folderPath, $zipPath) {
		$iterator = new \RecursiveIteratorIterator(
			new \RecursiveDirectoryIterator($folderPath, \RecursiveDirectoryIterator::SKIP_DOTS),
			\RecursiveIteratorIterator::SELF_FIRST);
		
		$zip->addEmptyDir($zipPath);
		
		foreach ($iterator as $item) {
			if ($item->isDir()) {
				$zip->addEmptyDir($zipPath . DIRECTORY_SEPARATOR . $iterator->getSubPathName());
			} else {
				$zip->addFile($item->getPathname(), $zipPath . DIRECTORY_SEPARATOR . $iterator->getSubPathName());
			}
		}
		
	}

	// Method to recursively delete a directory
	private function deleteDirectory($path) {
		if (is_dir($path)) {
			$iterator = new \RecursiveDirectoryIterator($path, \RecursiveDirectoryIterator::SKIP_DOTS);
			$files = new \RecursiveIteratorIterator($iterator, \RecursiveIteratorIterator::CHILD_FIRST);
			foreach ($files as $file) {
				if ($file->isDir()) {
					rmdir($file->getRealPath());
				} else {
					unlink($file->getRealPath());
				}
			}
			rmdir($path);
		}
	}

	// Method to handle restore functionality
	public function handleRestore($backupPostId) {
		// Code to restore a backup archive including modifying tables, copying content, etc.
		set_time_limit(0);

		// Get the current WordPress database prefix
		global $wpdb;
		$currentPrefix = $wpdb->prefix;

		// Generate a new prefix for the duplicated tables
		$newPrefix = $currentPrefix . 'clone_';

		// Duplicates all the tables connected to WordPress
		$this->duplicateTables($newPrefix);

		// Changes the prefix used for WordPress databases
		$wpdb->prefix = $newPrefix;
		$configPath = ABSPATH . 'wp-config.php';
		$this->changeDatabasePrefix($configPath, $newPrefix);

		// Delete the tables with the old prefix
		$this->deleteTables($currentPrefix, $newPrefix);
		
		// Get the backup archive to restore from plugin settings or user input
		$backupPath = get_post($backupPostId)->post_content;

		// Extract the backup archive to a temporary folder
		$tempPath = sys_get_temp_dir() . '/' . uniqid('tmp_');
		$this->extractBackupArchive($backupPath, $tempPath);

		// Restore the database
		$this->restoreDatabase($tempPath);

		// Create a backup of the website before restoring the previous version
		// Add the function here

		// Restore the WordPress content
		$this->restoreContent($tempPath);
		$wpdb->prefix = $currentPrefix;

		// Update the .htaccess and wp-config.php files
		//$this->updateFiles($tempPath);

		// Clean up the temporary folder
		$this->deleteDirectory($tempPath);

		// Display success message or perform any additional actions
		echo 'Backup restored successfully!';
	}

	// Changes the prefix used for WordPress databases
	function changeDatabasePrefix($configPath, $newPrefix) {
		// Read the contents of the wp-config.php file
		$configContent = file_get_contents($configPath);

		// Pattern to match the line containing $table_prefix
		$pattern = '/\$table_prefix\s+=\s+(\'|")(.+)(\'|")\s*;/i';

		// The line containing the new $table_prefix
		$replacement = '$table_prefix = \''.$newPrefix.'\';';

		// Replace the original prefix with the new prefix
		$newConfigContent = preg_replace($pattern, $replacement, $configContent);

		// Write the updated contents back to the wp-config.php file
		file_put_contents($configPath, $newConfigContent);

		return true; // Prefix changed successfully
	}

	public function duplicateTables($newPrefix) {
		global $wpdb;
		$currentPrefix = $wpdb->prefix;
	  
		// Retrieve the list of WordPress tables
		$tables = $wpdb->get_results("SHOW TABLES LIKE '{$currentPrefix}%'", ARRAY_N);

		// Iterate through each table and duplicate it
		foreach ($tables as $table) {
			$tableName = $table[0];
			$newTableName = str_replace($currentPrefix, $newPrefix, $tableName);

			// Duplicate the table using the duplicateTable() method
			$this->duplicateTable($tableName, $newTableName);
		}

		// Display success message or perform any additional actions
		echo 'Tables duplicated successfully!';
	}

	/**
	 * Duplicate a table with a new name
	 *
	 * @param string $tableName The name of the table to duplicate
	 * @param string $newTableName The new name for the duplicated table
	 */
	function duplicateTable($tableName, $newTableName) {
		global $wpdb;

		// Check if the table exists
		if ($wpdb->get_var("SHOW TABLES LIKE '$tableName'") === $tableName) {
			// Duplicate table structure
			$wpdb->query("CREATE TABLE $newTableName LIKE $tableName");

			// Duplicate table data
			$wpdb->query("INSERT $newTableName SELECT * FROM $tableName");
			
			echo "Table $tableName duplicated successfully as $newTableName!";
		} else {
			echo "Table $tableName does not exist!";
		}
	}

	// Delete a tables that start with a prefix
	function deleteTables($prefix, $keepPrefix) {
		global $wpdb;

		// Get the list of WordPress tables with the given prefix
		$tables = $wpdb->get_results("SHOW TABLES LIKE '$prefix%'");

		// Iterate through each table and drop it
		foreach ($tables as $table) {
			foreach ($table as $tableName) {
				if (!preg_match("/$keepPrefix.*/", $tableName))
					$wpdb->query("DROP TABLE IF EXISTS $tableName");
			}
		}
	}

	//Renames all of the tables connected and used by WordPress by adding an additional prefix to their name
	public function renameTables($oldPrefix, $newPrefix) {
		global $wpdb;

		// Retrieve the list of WordPress tables
		$tables = $wpdb->get_results($wpdb->prepare("SHOW TABLES LIKE '%s%'", $wpdb->prefix));

		// Iterate through each table and rename it
		foreach ($tables as $table) {
			foreach ($table as $tableName) {
				// Rename the table by adding the new prefix
				$newTableName = str_replace($oldPrefix, $newPrefix, $tableName);

				// Rename the table using the WordPress rename() method
				$wpdb->query($wpdb ->prepare("RENAME TABLE `%s` TO `%s`", $tableName, $newTableName));
			}
		}
	}

	// Method to restore the WordPress database
	private function restoreDatabase($tempPath) {
		global $wpdb;

		// Get the list of SQL backup files
		$sqlFiles = glob($tempPath . '/database/*.sql');
		
		// Iterate through each SQL file and restore the tables
		foreach ($sqlFiles as $sqlFile) {
			$tableName = basename($sqlFile, '.sql');
			
			// Load the SQL file and restore the table
			$sql = file_get_contents($sqlFile);
			
			// Drop the existing table if it exists
			$wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}{$tableName}");

			// Execute the SQL queries to create the table
			$pattern = "/\s*CREATE\s+TABLE\s+`\w+`\s*\((\s*[^,;]*,\s*)*\s*[^,;]*\s*\)[^;]*;/";
			preg_match($pattern, $sql, $createMatches);
			$wpdb->query($createMatches[0]);

			// Execute the SQL queries to restore the table
			$pattern = "/INSERT\s+INTO\s+(`\w+`)\s+\(.*\)\s+VALUES\s+\(.*\)\s*;/";
			preg_match_all($pattern, $sql, $insertMatches);
			foreach($insertMatches[0] as $match) {
				$wpdb->query($match);
			}
		}
	}

	// Method to restore the WordPress content
	private function restoreContent($tempPath) {
		// Get the path of the WordPress installation
		$wordpressPath = ABSPATH;
		
		// Restore the content by replacing the existing files with the backup files
		$this->copyDirectory($tempPath . '/wordpress', $wordpressPath);
	}

	// Method to update .htaccess and wp-config.php files
	private function updateFiles($tempPath) {
		// Get the path of the WordPress installation
		$wordpressPath = ABSPATH;

		// Update .htaccess file
		$htaccessPath = $wordpressPath . '.htaccess';

		// Check if .htaccess file exists
		if (file_exists($htaccessPath)) {
			$htaccessContent = file_get_contents($htaccessPath);
			
			// Update .htaccess content to redirect all requests to the newly created folder
			$newFolderName = basename($tempPath . '/content');
			$htaccessContent = preg_replace('/(RewriteRule .*)$/', "RewriteRule . /$newFolderName/index.php [L]", $htaccessContent);
			
			// Write updated .htaccess content back to the file
			file_put_contents($htaccessPath, $htaccessContent);
		}

		// Update wp-config.php file
		$configPath = $wordpressPath . 'wp-config.php';
		$configContent = file_get_contents($configPath);

		// Update wp-config.php content to add the table prefix used in the restored backup
		$tablePrefix = $wpdb->prefix;
		$configContent = preg_replace("/(table_prefix.*')('[^']*')/", "\$1'{$tablePrefix}\$2", $configContent);

		// Write updated wp-config.php content back to the file
		file_put_contents($configPath, $configContent);
	}

	// Method to extract the backup archive to a temporary folder
	private function extractBackupArchive($backupPath, $tempPath) {
		// Create a temporary folder to hold the extracted files
		mkdir($tempPath);

		// Extract the backup archive to the temporary folder
		$zip = new \ZipArchive();
		if ($zip->open($backupPath) === true) {
			$zip->extractTo($tempPath);
			$zip->close();
		}
	}

	// Method to handle cloning functionality
	public function handleClone() {
		// Code to create a backup and make it publicly available for cloning
		// Create a backup archive for cloning
		/*$this->handleBackup();

		// Get the backup archive to clone from plugin settings or user input
		$backupArchive = $_POST['backup_archive'];

		// Get the backup location from plugin settings
		$backupLocation = get_option('cbm_backup_location');
		$backupPath = $backupLocation . '/' . $backupArchive;

		// Make the backup archive publicly available
		$this->makeBackupPublic($backupPath);

		// Display success message or perform any additional actions
		echo 'Backup archive is now publicly available for cloning!';*/
	}

	// Method to make the backup archive publicly available
	private function makeBackupPublic($backupPath) {
		// Remove any previous access restrictions
		remove_all_filters('apache_mod_rewrite_rules');

		// Modify the .htaccess file to allow access only to a designated domain
		$htaccessPath = ABSPATH . '.htaccess';
		$backupUrl = home_url('?cbm_backup=' . basename($backupPath));

		// Check if .htaccess file exists
		if (file_exists($htaccessPath)) {
			$htaccessContent = file_get_contents($htaccessPath);
			
			// Add the necessary rules to restrict access to the backup archive
			$htaccessContent .= "\n\n#Restrict access to backup archive\n";
			$htaccessContent .= "RewriteCond %{REQUEST_URI} !(^|/)wp-admin/ [NC]\n";
			$htaccessContent .= "RewriteCond %{QUERY_STRING} cbm_backup=(.*)$\n";
			$htaccessContent .= "RewriteCond %{HTTP_REFERER} !^https?://(www\.)?example\.com [NC]\n"; // Replace with your backup domain
			$htaccessContent .= "RewriteRule ^(.*)$ - [F,L]\n";

			// Write updated .htaccess content back to the file
			file_put_contents($htaccessPath, $htaccessContent);
		}
	}

	// Method to handle the public access of the backup archive
	public function handlePublicAccess() {
		if (isset($_GET['cbm_backup'])) {
			$backupFilename = $_GET['cbm_backup'];
			$backupLocation = get_option('cbm_backup_location');
			$backupPath = $backupLocation . '/' . $backupFilename;

			// Set the appropriate content headers
			header('Content-Type: application/octet-stream');
			header('Content-Disposition: attachment; filename=' . $backupFilename);
			header('Content-Length: ' . filesize($backupPath));

			// Output the backup file
			readfile($backupPath);

			// Stop further script execution
			exit;
		}
	}

// Additional methods for handling other functionalities, such as archiving, importing, etc.

}